; 2nd Fixed version of _ClipBoard_GetData

; #FUNCTION# ====================================================================================================================
; Name...........: _ClipBoard_GetData
; Description ...: Retrieves data from the clipboard in a specified format
; Syntax.........: _ClipBoard_GetData([$iFormat = 1])
; Parameters ....: $iFormat     - Specifies a clipboard format:
;                  |$CF_TEXT            - Text format
;                  |$CF_BITMAP          - Handle to a bitmap (HBITMAP)
;                  |$CF_METAFILEPICT    - Handle to a metafile picture (METAFILEPICT)
;                  |$CF_SYLK            - Microsoft Symbolic Link (SYLK) format
;                  |$CF_DIF             - Software Arts' Data Interchange Format
;                  |$CF_TIFF            - Tagged image file format
;                  |$CF_OEMTEXT         - Text format containing characters in the OEM character set
;                  |$CF_DIB             - BITMAPINFO structure followed by the bitmap bits
;                  |$CF_PALETTE         - Handle to a color palette
;                  |$CF_PENDATA         - Data for the pen extensions to Pen Computing
;                  |$CF_RIFF            - Represents audio data in RIFF format
;                  |$CF_WAVE            - Represents audio data in WAVE format
;                  |$CF_UNICODETEXT     - Unicode text format
;                  |$CF_ENHMETAFILE     - Handle to an enhanced metafile (HENHMETAFILE)
;                  |$CF_HDROP           - Handle to type HDROP that identifies a list of files
;                  |$CF_LOCALE          - Handle to the locale identifier associated with text in the clipboard
;                  |$CF_DIBV5           - BITMAPV5HEADER structure followed by bitmap color and the bitmap bits
;                  |$CF_OWNERDISPLAY    - Owner display format
;                  |$CF_DSPTEXT         - Text display format associated with a private format
;                  |$CF_DSPBITMAP       - Bitmap display format associated with a private format
;                  |$CF_DSPMETAFILEPICT - Metafile picture display format associated with a private format
;                  |$CF_DSPENHMETAFILE  - Enhanced metafile display format associated with a private format
; Return values .: Success      - Text for text based formats or Binary data for all other formats
;                               -  @extended is set to the # of characters for Text, or # of bytes for Binary
;                  Failure      - 0
; Author ........: Paul Campbell (PaulIA)
; Modified.......: Gary Frost,
;                  Ascend4nt (now follows traditional ClipBoard 'get' code, fixed $CF_UNICODETEXT errors)
; Remarks .......: This function performs all of the steps neccesary to get data from the clipboard. It checks to see if 
;                  the data format is available, opens the clipboard, closes the clipboard and returns the data in one of
;                  two formats: String format for datatypes $CF_TEXT, $CF_OEMTEXT, or $CF_UNICODETEXT,
;                  or Binary format for every other type. If you need a finer degree of control over retrieving data from
;                  the clipboard, you may want to use the _ClipBoard_GetDataEx function. 
; Related .......: _ClipBoard_GetDataEx, _ClipBoard_SetData
; Link ..........;
; Example .......; Yes
; ===============================================================================================================================
Func _ClipBoard_GetData($iFormat = 1)
	Local $hMemory, $tData, $pMemoryBlock, $iDataSize, $vReturn
	
	; Reset Global structure, releast memory
	$_stClipGetStruct = 0

	If Not _ClipBoard_IsFormatAvailable($iFormat) Then Return SetError(-1, 0, 0)
	If Not _ClipBoard_Open(0) Then Return SetError(-2, 0, 0)
	$hMemory = _ClipBoard_GetDataEx($iFormat)

	;_ClipBoard_Close()		; moved to end: traditionally done *after* copying over the memory

	If $hMemory=0 Then
		_ClipBoard_Close()
		Return SetError(-3, 0, 0)
	EndIf
	
	Local $pMemoryBlock=_MemGlobalLock($hMemory)
	
	If $pMemoryBlock=0 Then
		_ClipBoard_Close()
		Return SetError(-4,0,0)
	EndIf
	
	; Get the actual memory size of the ClipBoard memory object (in bytes)
	$iDataSize=_MemGlobalSize($hMemory)
	
	If $iDataSize = 0 Then
		_MemGlobalUnlock($hMemory)
		_ClipBoard_Close()
		Return SetError(-5,0,"")
	EndIf

	Switch $iFormat
		Case $CF_TEXT, $CF_OEMTEXT
			$tData = DllStructCreate("char[" & $iDataSize & "]", $pMemoryBlock)
		Case $CF_UNICODETEXT
			; Round() shouldn't be necessary, as CF_UNICODETEXT should be 2-bytes wide & thus evenly-divisible
			$iDataSize=Round($iDataSize/2)
			$tData = DllStructCreate("wchar[" & $iDataSize & "]", $pMemoryBlock)
		Case Else
			; Binary data return for all other formats
			$tData = DllStructCreate("ubyte[" & $iDataSize & "]", $pMemoryBlock)
	EndSwitch
	; Grab the data from the Structure so the Memory can be unlocked
	$vReturn = DllStructGetData($tData, 1)
	
	; Unlock the memory & Close the clipboard now that we have grabbed what we needed
	_MemGlobalUnlock($hMemory)
	_ClipBoard_Close()
	
	; Return the size of the string or binary object in @extended
	SetError(0,$iDataSize)
	Return $vReturn
EndFunc   ;==>_ClipBoard_GetData
